function N = count(img);
% N = count(img)
%
% Count the number of fluorescent beads in an image. Initial detection of
% bead areas is done by thresholding the image by Otsu's method. Individual
% beads are detected from clumps of beads by a seeded watershed method.
% Local maxima of the original bead image are used as seeds.
%
% In:
%  img     16-bit grayscale image
%
% Out:
%  N       number of beads in the image
%
% Antti Niemist 2009/12/15

% Adjust the image such that the pixel values span [0,1]
datatype = 'uint16';
img = double(img)/double(intmax(datatype));
img = imadjust(img,[min(img(:)),max(img(:))]);

% Filter the image with a 5x5 median filter
img = medfilt2(img,[5 5],'symmetric');

% Threshold the image with a threshold found by Otsu's method
T = graythresh(img);
bw = im2bw(img,T);

% Return error code -1 if there is too much foreground (five percent would
% be equivalent to ~30 beads)
if sum(bw(:)==1)/prod(size(bw)) > .05
  N = -1;
  return
end

% Separate touching beads by seeded watershedcut.
seeds = flocmaxima(img,5,bw);
bw_b = seededwatershedcut(bw,seeds);

% Fill holes in objects
se = strel('disk',2);
bw_c = imfill(bw_b,'holes');

% Remove small foreground objects
minsize = 100;
bw_d = bwareaopen(bw_c,minsize);

% Label the last two binary images and count the number of objects
% detected in each
[dummy,N_c] = bwlabel(bw_c);
[lab,N] = bwlabel(bw_d);

% Return error code -2 if many small objects got removed
if N_c-N >= 5
  N = -2;
  return
end

% Find object areas
stats = regionprops(lab,'Area');

% Return error code -3 if object sizes vary more than by a factor of 2
area_max = max([stats.Area]);
area_min = min([stats.Area]);
if area_max>2*area_min
  N = -3;
  return
end

% Return error code -4 if any of the objects is not round according to a
% form factor criterion
ff = formfactor(bw_d);
if min(ff) < .6
  N = -4;
  return
end
